/*
 * Decompiled with CFR 0.152.
 */
package com.easywebmap.map;

import java.io.IOException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

public class DiskTileCache {
    private final Path cacheDirectory;
    private final ConcurrentHashMap<String, Long> tileTimestamps;
    private final ExecutorService diskExecutor;

    public DiskTileCache(Path dataDirectory) {
        this.cacheDirectory = dataDirectory.resolve("tilecache");
        this.tileTimestamps = new ConcurrentHashMap();
        this.diskExecutor = Executors.newFixedThreadPool(6, r -> {
            Thread t = new Thread(r, "EasyWebMap-DiskIO");
            t.setDaemon(true);
            return t;
        });
        try {
            Files.createDirectories(this.cacheDirectory, new FileAttribute[0]);
        }
        catch (IOException e) {
            System.err.println("[EasyWebMap] Failed to create tile cache directory: " + e.getMessage());
        }
    }

    public byte[] get(String worldName, int zoom, int x, int z) {
        Path tilePath = this.getTilePath(worldName, zoom, x, z);
        if (!Files.exists(tilePath, new LinkOption[0])) {
            return null;
        }
        try {
            return Files.readAllBytes(tilePath);
        }
        catch (IOException e) {
            return null;
        }
    }

    public CompletableFuture<byte[]> getAsync(String worldName, int zoom, int x, int z) {
        return CompletableFuture.supplyAsync(() -> this.get(worldName, zoom, x, z), this.diskExecutor);
    }

    public void putAsync(String worldName, int zoom, int x, int z, byte[] data) {
        String key = this.createKey(worldName, zoom, x, z);
        this.tileTimestamps.put(key, System.currentTimeMillis());
        this.diskExecutor.execute(() -> this.putSync(worldName, zoom, x, z, data));
    }

    public void put(String worldName, int zoom, int x, int z, byte[] data) {
        this.putSync(worldName, zoom, x, z, data);
        String key = this.createKey(worldName, zoom, x, z);
        this.tileTimestamps.put(key, System.currentTimeMillis());
    }

    private void putSync(String worldName, int zoom, int x, int z, byte[] data) {
        Path tilePath = this.getTilePath(worldName, zoom, x, z);
        try {
            Files.createDirectories(tilePath.getParent(), new FileAttribute[0]);
            Files.write(tilePath, data, new OpenOption[0]);
        }
        catch (IOException e) {
            System.err.println("[EasyWebMap] Failed to cache tile: " + e.getMessage());
        }
    }

    public long getTileAge(String worldName, int zoom, int x, int z) {
        String key = this.createKey(worldName, zoom, x, z);
        Long cachedTimestamp = this.tileTimestamps.get(key);
        if (cachedTimestamp != null) {
            return System.currentTimeMillis() - cachedTimestamp;
        }
        Path tilePath = this.getTilePath(worldName, zoom, x, z);
        if (!Files.exists(tilePath, new LinkOption[0])) {
            return Long.MAX_VALUE;
        }
        try {
            BasicFileAttributes attrs = Files.readAttributes(tilePath, BasicFileAttributes.class, new LinkOption[0]);
            long fileTime = attrs.lastModifiedTime().toMillis();
            this.tileTimestamps.put(key, fileTime);
            return System.currentTimeMillis() - fileTime;
        }
        catch (IOException e) {
            return Long.MAX_VALUE;
        }
    }

    public boolean exists(String worldName, int zoom, int x, int z) {
        return Files.exists(this.getTilePath(worldName, zoom, x, z), new LinkOption[0]);
    }

    public void clear() {
        this.tileTimestamps.clear();
        try {
            if (Files.exists(this.cacheDirectory, new LinkOption[0])) {
                Files.walk(this.cacheDirectory, new FileVisitOption[0]).sorted((a, b) -> b.compareTo((Path)a)).forEach(path -> {
                    try {
                        Files.delete(path);
                    }
                    catch (IOException iOException) {
                        // empty catch block
                    }
                });
                Files.createDirectories(this.cacheDirectory, new FileAttribute[0]);
            }
        }
        catch (IOException e) {
            System.err.println("[EasyWebMap] Failed to clear tile cache: " + e.getMessage());
        }
    }

    public void clearWorld(String worldName) {
        Path worldDir = this.cacheDirectory.resolve(worldName);
        if (!Files.exists(worldDir, new LinkOption[0])) {
            return;
        }
        try {
            Files.walk(worldDir, new FileVisitOption[0]).sorted((a, b) -> b.compareTo((Path)a)).forEach(path -> {
                try {
                    Files.delete(path);
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            });
        }
        catch (IOException e) {
            System.err.println("[EasyWebMap] Failed to clear world cache: " + e.getMessage());
        }
        this.tileTimestamps.entrySet().removeIf(entry -> ((String)entry.getKey()).startsWith(worldName + "/"));
    }

    public void shutdown() {
        this.diskExecutor.shutdown();
    }

    private Path getTilePath(String worldName, int zoom, int x, int z) {
        return this.cacheDirectory.resolve(worldName).resolve(String.valueOf(zoom)).resolve(x + "_" + z + ".png");
    }

    private String createKey(String worldName, int zoom, int x, int z) {
        return worldName + "/" + zoom + "/" + x + "/" + z;
    }

    public Path getCacheDirectory() {
        return this.cacheDirectory;
    }
}

